/***************************************************************************/
/* VESA Generalized Timing Formula(GTF) calculator for VESA VBE 3.0 Driver */
/***************************************************************************/
/*        Last updated: 27.2.2011 [DJGPP]                                  */
/*        Copyright (C) 2000-2011 SciTech Software                         */
/*        Modified output for VGA13/VESADRV library by Martin Rehak        */
/*        Contact: rayer@seznam.cz, rehakm2@feld.cvut.cz                   */
/***************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <ctype.h>
#include <math.h>

//********************* types ***********************************************
#define CFGFN     "CRTC.CFG"
#define false      0
#define true       1
typedef int        ibool;              // Integer boolean type

/* Define the structures for holding the horizontal and vertical
 * CRTC parameters for a mode.
 *
 * Note: The sync timings are defined in both VGA compatible timings
 * (sync start and sync end positions) and also in GTF compatible
 * modes with the front porch, sync width and back porch defined.
 */

typedef struct {
  int hTotal;                          // Horizontal total
  int hDisp;                           // Horizontal displayed
  int hSyncStart;                      // Horizontal sync start
  int hSyncEnd;                        // Horizontal sync end
  int hFrontPorch;                     // Horizontal front porch
  int hSyncWidth;                      // Horizontal sync width
  int hBackPorch;                      // Horizontal back porch
} GTF_hCRTC;

typedef struct {
  int vTotal;                          // Vertical total
  int vDisp;                           // Vertical displayed
  int vSyncStart;                      // Vertical sync start
  int vSyncEnd;                        // Vertical sync end
  int vFrontPorch;                     // Vertical front porch
  int vSyncWidth;                      // Vertical sync width
  int vBackPorch;                      // Vertical back porch
} GTF_vCRTC;

// Define the main structure for holding generated GTF timings
typedef struct {
  GTF_hCRTC h;                         // Horizontal CRTC paremeters
  GTF_vCRTC v;                         // Vertical CRTC parameters
  char      hSyncPol;                  // Horizontal sync polarity
  char      vSyncPol;                  // Vertical sync polarity
  char      interlace;                 // 'I' for Interlace, 'N' for Non
  double    vFreq;                     // Vertical frequency (Hz)
  double    hFreq;                     // Horizontal frequency (KHz)
  double    dotClock;                  // Pixel clock (Mhz)
} GTF_timings;

// Define the structure for holding standard GTF formula constants
typedef struct {
  double margin;                       // Margin size as percentage of display
  double cellGran;                     // Character cell granularity
  double minPorch;                     // Minimum front porch in lines/chars
  double vSyncRqd;                     // Width of V sync in lines
  double hSync;                        // Width of H sync as percent of total
  double minVSyncBP;                   // Minimum vertical sync + back porch (us)
  double m;                            // Blanking formula gradient
  double c;                            // Blanking formula offset
  double k;                            // Blanking formula scaling factor
  double j;                            // Blanking formula scaling factor weight
} GTF_constants;

#define GTF_lockVF 1                   // Lock to vertical frequency
#define GTF_lockHF 2                   // Lock to horizontal frequency
#define GTF_lockPF 3                   // Lock to pixel clock frequency

//------------------------- Global Variables --------------------------------
static GTF_constants GC = {
  1.8,                                 // Margin size as percentage of display
  8,                                   // Character cell granularity
  1,                                   // Minimum front porch in lines/chars
  3,                                   // Width of V sync in lines
  8,                                   // Width of H sync as percent of total
  550,                                 // Minimum vertical sync + back porch (us)
  600,                                 // Blanking formula gradient
  40,                                  // Blanking formula offset
  128,                                 // Blanking formula scaling factor
  20,                                  // Blanking formula scaling factor weight
  };

//-------------------------- Implementation ---------------------------------
static double round(double v)
{
  return floor(v + 0.5);
}

static void GetInternalConstants(GTF_constants *c)
/****************************************************************************
*                                                                           *
* Function:     GetInternalConstants                                        *
* Parameters:   c - Place to store the internal constants                   *
*                                                                           *
* Description:  Calculates the rounded, internal set of GTF constants.      *
*               These constants are different to the real GTF constants     *
*               that can be set up for the monitor. The calculations to     *
*               get these real constants are defined in the 'Work Area'     *
*               after the constants are defined in the Excel spreadsheet.   *
*                                                                           *
****************************************************************************/
{
  c->margin = GC.margin;
  c->cellGran = round(GC.cellGran);
  c->minPorch = round(GC.minPorch);
  c->vSyncRqd = round(GC.vSyncRqd);
  c->hSync = GC.hSync;
  c->minVSyncBP = GC.minVSyncBP;
  if (GC.k == 0)
    c->k = 0.001;
  else
    c->k = GC.k;
  c->m = (c->k / 256) * GC.m;
  c->c = (GC.c - GC.j) * (c->k / 256) + GC.j;
  c->j = GC.j;
}

void GTF_calcTimings(double hPixels,double vLines,double freq, int type,ibool wantMargins,ibool wantInterlace,GTF_timings *t)
/****************************************************************************
*                                                                           *
* Function:     GTF_calcTimings                                             *
* Parameters:   hPixels  - X resolution                                     *
*               vLines   - Y resolution                                     *
*               freq     - Frequency (Hz, KHz or MHz depending on type)     *
*               type     - 1 - vertical, 2 - horizontal, 3 - dot clock      *
*               margins  - True if margins should be generated              *
*               interlace - True if interlaced timings to be generated      *
*               t        - Place to store the resulting timings             *
*                                                                           *
* Description:  Calculates a set of GTF timing parameters given a specified *
*               resolution and vertical frequency. The horizontal frequency *
*               and dot clock will be automatically generated by this       *
*               routines.                                                   *
*                                                                           *
*               For interlaced modes the CRTC parameters are calculated for *
*               a single field, so will be half what would be used in       *
*               a non-interlaced mode.                                      *
*                                                                           *
****************************************************************************/
{
  double interlace, vFieldRate, hPeriod=0;
  double topMarginLines, botMarginLines;
  double leftMarginPixels, rightMarginPixels;
  double hPeriodEst=0, vSyncBP=0, vBackPorch=0;
  double vTotalLines=0, vFieldRateEst;
  double hTotalPixels, hTotalActivePixels, hBlankPixels;
  double idealDutyCycle=0, hSyncWidth, hSyncBP, hBackPorch;
  double idealHPeriod;
  double vFreq, hFreq, dotClock;
  GTF_constants c;

  // Get rounded GTF constants used for internal calculations
  GetInternalConstants(&c);

  // Move input parameters into appropriate variables
  vFreq = hFreq = dotClock = freq;

  // Round pixels to character cell granularity
  hPixels = round(hPixels / c.cellGran) * c.cellGran;

  // For interlaced mode halve the vertical parameters, and double the required field refresh rate.
  vFieldRate = vFreq;
  interlace = 0;
  if (wantInterlace)
    dotClock *= 2;

  // Determine the lines for margins
  if (wantMargins)
    {
    topMarginLines = round(c.margin / 100 * vLines);
    botMarginLines = round(c.margin / 100 * vLines);
    }
  else
    {
    topMarginLines = 0;
    botMarginLines = 0;
    }

  if (type != GTF_lockPF)
    {
    if (type == GTF_lockVF)
      {
      // Estimate the horizontal period
      hPeriodEst = ((1/vFieldRate)-(c.minVSyncBP/1000000))/(vLines+(2*topMarginLines)+c.minPorch+interlace)*1000000;
      // Find the number of lines in vSync + back porch
      vSyncBP = round(c.minVSyncBP / hPeriodEst);
      }
    else if (type == GTF_lockHF)
      {
      // Find the number of lines in vSync + back porch
      vSyncBP = round((c.minVSyncBP * hFreq) / 1000);
      }

  // Find the number of lines in the V back porch alone
  vBackPorch = vSyncBP - c.vSyncRqd;

  // Find the total number of lines in the vertical period
  vTotalLines = vLines + topMarginLines + botMarginLines + vSyncBP + interlace + c.minPorch;

  if (type == GTF_lockVF)
    {
    // Estimate the vertical frequency
    vFieldRateEst = 1000000 / (hPeriodEst * vTotalLines);

    // Find the actual horizontal period
    hPeriod = (hPeriodEst * vFieldRateEst) / vFieldRate;

    // Find the actual vertical field frequency
    vFieldRate = 1000000 / (hPeriod * vTotalLines);
    }
  else if (type == GTF_lockHF)
    {
    // Find the actual vertical field frequency
    vFieldRate = (hFreq / vTotalLines) * 1000;
    }
  }

  // Find the number of pixels in the left and right margins
  if (wantMargins)
    {
    leftMarginPixels = round(hPixels * c.margin) / (100 * c.cellGran);
    rightMarginPixels = round(hPixels * c.margin) / (100 * c.cellGran);
    }
  else
    {
    leftMarginPixels = 0;
    rightMarginPixels = 0;
    }

  // Find the total number of active pixels in image + margins
  hTotalActivePixels = hPixels + leftMarginPixels + rightMarginPixels;

  if (type == GTF_lockVF)
    {
    // Find the ideal blanking duty cycle
    idealDutyCycle = c.c - ((c.m * hPeriod) / 1000);
    }
  else if (type == GTF_lockHF)
    {
    // Find the ideal blanking duty cycle
    idealDutyCycle = c.c - (c.m / hFreq);
    }
  else if (type == GTF_lockPF)
    {
    // Find ideal horizontal period from blanking duty cycle formula
    idealHPeriod = (((c.c-100)+(sqrt((pow(100-c.c,2))+(0.4*c.m*(hTotalActivePixels+rightMarginPixels+leftMarginPixels)/dotClock))))/(2*c.m))*1000;
    // Find the ideal blanking duty cycle
    idealDutyCycle = c.c - ((c.m * idealHPeriod) / 1000);
    }

  // Find the number of pixels in blanking time
  hBlankPixels = round((hTotalActivePixels * idealDutyCycle) / ((100 - idealDutyCycle) * c.cellGran)) * c.cellGran;

  // Find the total number of pixels
  hTotalPixels = hTotalActivePixels + hBlankPixels;

  // Find the horizontal back porch
  hBackPorch = round((hBlankPixels / 2) / c.cellGran) * c.cellGran;

  // Find the horizontal sync width
  hSyncWidth = round(((c.hSync/100) * hTotalPixels) / c.cellGran) * c.cellGran;

  // Find the horizontal sync + back porch
  hSyncBP = hBackPorch + hSyncWidth;

  if (type == GTF_lockPF)
    {
    // Find the horizontal frequency
    hFreq = (dotClock / hTotalPixels) * 1000;

    // Find the number of lines in vSync + back porch
    vSyncBP = round((c.minVSyncBP * hFreq) / 1000);

    // Find the number of lines in the V back porch alone
    vBackPorch = vSyncBP - c.vSyncRqd;

    // Find the total number of lines in the vertical period
    vTotalLines = vLines + topMarginLines + botMarginLines + vSyncBP
        + interlace + c.minPorch;

    // Find the actual vertical field frequency
        vFieldRate = (hFreq / vTotalLines) * 1000;
    }
  else
    {
    if (type == GTF_lockVF)
      {
      // Find the horizontal frequency
      hFreq = 1000 / hPeriod;
      }
    else if (type == GTF_lockHF)
      {
      // Find the horizontal frequency
      hPeriod = 1000 / hFreq;
      }

    // Find the pixel clock frequency
    dotClock = hTotalPixels / hPeriod;
    }

  // Return the computed frequencies
  t->vFreq = vFieldRate;
  t->hFreq = hFreq;
  t->dotClock = dotClock;

  // Determine the vertical timing parameters
  t->h.hTotal = (int)hTotalPixels;
  t->h.hDisp = (int)hTotalActivePixels;
  t->h.hSyncStart = t->h.hTotal - (int)hSyncBP;
  t->h.hSyncEnd = t->h.hTotal - (int)hBackPorch;
  t->h.hFrontPorch = t->h.hSyncStart - t->h.hDisp;
  t->h.hSyncWidth = (int)hSyncWidth;
  t->h.hBackPorch = (int)hBackPorch;

  // Determine the vertical timing parameters
  t->v.vTotal = (int)vTotalLines;
  t->v.vDisp = (int)vLines;
  t->v.vSyncStart = t->v.vTotal - (int)vSyncBP;
  t->v.vSyncEnd = t->v.vTotal - (int)vBackPorch;
  t->v.vFrontPorch = t->v.vSyncStart - t->v.vDisp;
  t->v.vSyncWidth = (int)c.vSyncRqd;
  t->v.vBackPorch = (int)vBackPorch;
  if (wantInterlace)
    {
    // Halve the timings for interlaced modes
    t->v.vTotal /= 2;
    t->v.vDisp /= 2;
    t->v.vSyncStart /= 2;
    t->v.vSyncEnd /= 2;
    t->v.vFrontPorch /= 2;
    t->v.vSyncWidth /= 2;
    t->v.vBackPorch /= 2;
    t->dotClock /= 2;
    }

  // Mark as GTF timing using the sync polarities
  t->interlace = (wantInterlace) ? 'I' : 'N';
  t->hSyncPol = '-';
  t->vSyncPol = '+';
}

void GTF_getConstants(GTF_constants *constants)
{ *constants = GC; }

void GTF_setConstants(GTF_constants *constants)
{ GC = *constants; }

int main(int argc, char *argv[])
{
  FILE *f;
  double xPixels, yPixels, freq;
  ibool interlace;
  GTF_timings t;
  // Print usage
  puts("\nGTFCALC 1.3 (C) 2000-2011 by SciTech Software & Martin Rehak; rayer@seznam.cz");
  printf("Compiled by GCC %s at %s, %s\n", __VERSION__, __TIME__, __DATE__);
  if ((argc!=5) && (argc!=6))
    {
    puts("Usage: GTFCALC <xPixels> <yPixels> <freq> [[Hz] [KHz] [MHz]] [I]\n");
    puts("where <xPixels> is the horizontal resolution of the mode, <yPixels> is the");
    puts("vertical resolution of the mode. The <freq> value will be the frequency to");
    puts("drive the calculations, and will be either the vertical frequency (in Hz)");
    puts("the horizontal frequency (in KHz) or the dot clock (in MHz). To generate");
    puts("timings for an interlaced mode, add 'I' to the end of the command line.\n");
    puts("For example to generate timings for 640x480 at 60Hz vertical:");
    puts("    GTFCALC 640 480 60 Hz\n");
    puts("For example to generate timings for 640x480 at 31.5KHz horizontal:");
    puts("    GTFCALC 640 480 31.5 KHz\n");
    puts("For example to generate timings for 640x480 with a 25.175Mhz dot clock:");
    puts("    GTFCALC 640 480 25.175 MHz\n");
    puts("GTFCALC will print the summary of calculated results and dump the CRTC");
    puts("values to CRTC.CFG config file in the format used by Martin's VGA13/VESA driver");
    puts("and SciTech Display Doctor.");
    exit(-1);
    }

  // Get values from command line
  xPixels = atof(argv[1]);
  yPixels = atof(argv[2]);
  freq = atof(argv[3]);
  interlace = ((argc == 6) && (toupper(argv[5][0]) == 'I'));

  // Compute the CRTC timings
  if (toupper(argv[4][0]) == 'H')
    GTF_calcTimings(xPixels, yPixels, freq, GTF_lockVF, false, interlace, &t);
  else if (toupper(argv[4][0]) == 'K')
    GTF_calcTimings(xPixels, yPixels, freq, GTF_lockHF, false, interlace, &t);
  else if (toupper(argv[4][0]) == 'M')
    GTF_calcTimings(xPixels, yPixels, freq, GTF_lockPF, false, interlace, &t);
  else {
    puts("Unknown command line!");
    exit(-2);
    }

  // Dump summary info to standard output
  printf("CRTC values for %.0fx%.0f @ %.2f %s\n\n", xPixels, yPixels, freq, argv[4]);
  printf("  hTotal      = %-4d    vTotal      = %-4d\n",t.h.hTotal, t.v.vTotal);
  printf("  hDisp       = %-4d    vDisp       = %-4d\n",t.h.hDisp, t.v.vDisp);
  printf("  hSyncStart  = %-4d    vSyncStart  = %-4d\n",t.h.hSyncStart, t.v.vSyncStart);
  printf("  hSyncEnd    = %-4d    vSyncEnd    = %-4d\n",t.h.hSyncEnd, t.v.vSyncEnd);
  printf("  hFrontPorch = %-4d    vFrontPorch = %-4d\n",t.h.hFrontPorch, t.v.vFrontPorch);
  printf("  hSyncWidth  = %-4d    vSyncWidth  = %-4d\n",t.h.hSyncWidth, t.v.vSyncWidth);
  printf("  hBackPorch  = %-4d    vBackPorch  = %-4d\n\n",t.h.hBackPorch, t.v.vBackPorch);
  printf("  Interlaced  = %s\n", (t.interlace == 'I') ? "Yes" : "No");
  printf("  H sync pol  = %c\n", t.hSyncPol);
  printf("  V sync pol  = %c\n\n", t.vSyncPol);
  printf("  Vert freq   = %.2f Hz\n", t.vFreq);
  printf("  Horiz freq  = %.2f KHz\n", t.hFreq);
  printf("  Dot Clock   = %.2f Mhz\n", t.dotClock);

  // Check for existing config file
  if ((f=fopen(CFGFN, "r"))!=NULL)
    {
    fclose(f);
    printf("\nWARNING: file %s already exist, overwrite it? [y/n]\n", CFGFN);
    if (toupper(getch())=='N')
      exit(-3);
    }

  if ((f=fopen(CFGFN, "w"))!=NULL)
    {
    fprintf(f, "[%.0f %.0f]\n", xPixels, yPixels);
    fprintf(f, "%d %d %d %d '%c' %s\n", t.h.hTotal, t.h.hDisp, t.h.hSyncStart, t.h.hSyncEnd, t.hSyncPol, (t.interlace == 'I') ? "I" : "NI");
    fprintf(f, "%d %d %d %d '%c'\n", t.v.vTotal, t.v.vDisp, t.v.vSyncStart, t.v.vSyncEnd, t.vSyncPol);
    fprintf(f, "%.2f\n", t.dotClock);
    fprintf(f, "# end of GTFCALC configuration data\n\n");
    fprintf(f, "# set calculated CRTC data for current videomode, for calc. run GTFCALC program [0/1]\n");
    fprintf(f, "set_crtc = 1\n");      // default accept CRTC values
    fprintf(f, "# bits per pixel [8/15/16/24/32]\n");
    fprintf(f, "bpp = 32\n");          // default bpp=32
    fprintf(f, "# use Linear Frame Buffer mode if available (on VESA VBE 2.0+ adapters) [0/1], set 0 if run under Windows NT\n");
    fprintf(f, "use_lfb = 1\n");       // use LFB
    fprintf(f, "# use MTRR Write-Combining memory mode for LFB/BS window on P6+/K7+ systems [0/1]\n");
    fprintf(f, "use_mtrr_wc = 0\n");   // allow MTRR WC set (works only on P6)
    fprintf(f, "# use vertical synchronization [0/1]\n");
    fprintf(f, "use_vsync = 1\n");     // allow vsync
    fclose(f);
    printf("\nfile %s was successfully created.\n", CFGFN);
    }
  else
    {
    printf("\nERROR: cannot create file %s\n", CFGFN);
    exit(-4);
    }
  return(0);
}
